/* FILE: mfile.c                                (D. Tottingham  04/28/91)

This is a collection of C functions that manage the waveform files for xdetect.
All functions have been written and compiled medium model.  The following
functions are included:

f_check_pathname ()             check pathname for validity
f_get_event_number ()           get current event number
f_get_pathname ()               get pathname
f_initialize_index ()           initialize the index file
f_initialize_params ()          initialize key parameters
f_set_pathname ()               set pathname
f_set_NetworkNodeId ()          set network node id
f_write_buffers ()              write data buffer(s) to disk

EXTERNAL FUNCTIONS CALLED:

b_convert_base10 ()             convert base 36 number into base 10 number
b_incr_base36 ()                increment a base 36 number
b_init_base36 ()                initialize a base 36 number
dm_get_current_time ()          get current time
dm_get_first_buffer ()          get first new demux buffer from demux queue
dm_get_head_buffer ()           get buffer at head of queue
dm_get_next_buffer ()           get next buffer from queue
dm_get_tail_buffer ()           get buffer at tail of demux queue
dsp_get_spectral_channel ()     get the spectral channel
er_abort ()                     display an error message then quit
h_increment_event_ctr ()        increment event counter
h_initialize_path ()            initialize path_name
h_set_event_ctr ()              set event counter
h_update ()                     update the header
l_get_location ()               get location if event was located
o_write_logfile ()              write string to log file stream
pk_get_first_arrival ()         get first p-arrival
pk_get_first_magnitude ()       get first magnitude
pk_get_next_arrival ()          get next p-arrival
pk_get_next_magnitude ()        get next magnitude
st_get_head_station ()          get station at head of station queue
st_get_next_station ()          get next station from station queue
suds_initialize ()              initialize a suds structure
suds_initialize_tag ()          initialize a suds structtag
t_get_head_trigger ()           get trigger information at head of channel queue
t_get_last_trigger ()           get last trigger
t_get_next_trigger ()           get next trigger from channel queue
u_build_date ()                 build a date: YRMNDY
u_build_date_fn ()              build a date filename: pathname\YRMNDY
u_get_diskspace ()              get the percentage of free disk space
u_strncpy ()                    copies n characters of far string2 to far string1

HISTORY:
   none

*/


/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <dos.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "mbase36.h"
#include "mconst.h"
#include "mdosfunc.h"
#include "mdemux.h"
#include "mdsp.h"
#include "mdt28xx.h"
#include "merror.h"
#include "mfile.h"
#include "mlocate.h"
#include "mlog.h"
#include "mpick.h"
#include "mscrnhdr.h"
#include "mstation.h"
#include "msudsini.h"
#include "mtrigger.h"
#include "mutils.h"
#include "xdetect.h"
#include "timer.h"


/*************************************************************************
                                GLOBALS

*************************************************************************/
PRIVATE F_HEADER far file_header;
PRIVATE char demux_filename[MAX_FILENAME_LENGTH * 2];
PRIVATE char index_filename[MAX_FILENAME_LENGTH * 2];
PRIVATE char path_name[MAX_FILENAME_LENGTH];
PRIVATE int far index_buffer[INDEX_FILESIZE];
PRIVATE char far header_buffer[BLOCK_SIZE];
PRIVATE int current_offset;
PRIVATE B_BASE36 index;
PRIVATE long event_number;


/*=======================================================================*
 *                            build_index_fn                             *
 *=======================================================================*/
/* Build a index filename: pathname\INDEX.@@@                            */

PRIVATE
void build_index_fn ()
{
   sprintf (index_filename, "%s%s", path_name, INDEX_FILENAME);
}

/*=======================================================================*
 *                            build_calib_fn                             *
 *=======================================================================*/
/* Build a calibration filename: pathname\YRMNDYxx.CLy
   where xx is the calibration channel in hex, and y is the network code.*/

PRIVATE
int build_calib_fn (abs_time)
double abs_time;
{
   int j;

   j = u_build_date_fn (abs_time, path_name, demux_filename);

   j += sprintf (demux_filename+j, "%02x.CL%c", dsp_get_spectral_channel(),
                 file_header.info.net_node_id[0]);

   return (j);
}

/*=======================================================================*
 *                          build_event_number                           *
 *=======================================================================*/
/* Build a unique event number: YRMNDYxxx, where xxx is a three digit
   base10 number.                                                        */

PRIVATE
long build_event_number (abs_time)
double abs_time;
{
   long date, event_number;

   date = u_build_date (abs_time);

   event_number = (date * 1000) + b_convert_base10(index);
   return (event_number);
}

/*=======================================================================*
 *                            build_event_fn                             *
 *=======================================================================*/
/* Build an event filename: pathname\YRMNDYxx.WVy
   where xx is a two digit base36 number, and y is the network code.     */

PRIVATE
int build_event_fn (abs_time)
double abs_time;
{
   int j;

   j = u_build_date_fn (abs_time, path_name, demux_filename);

   j += sprintf (demux_filename+j, "%s%s.WV%c", base36_num[index.high],
                 base36_num[index.low], file_header.info.net_node_id[0]);

   return (j);
}

/*=======================================================================*
 *                             update_index                              *
 *=======================================================================*/
/* Increment the index and update the index file.                        */

PRIVATE
void update_index ()
{
   struct tm * local_tm;
   long time;
   int fd;

   time = (long) (dm_get_current_time());
   local_tm = localtime (&time);

   if ( index_buffer[DAY]   == local_tm->tm_mday  &&
        index_buffer[MONTH] == local_tm->tm_mon + 1  &&
        index_buffer[YEAR]  == local_tm->tm_year) {
      b_incr_base36 (&index);
      index_buffer[HIGH] = index.high;
      index_buffer[LOW] = index.low;
   } else {
      b_init_base36 (&index);
      b_incr_base36 (&index);
      index_buffer[DAY] = local_tm->tm_mday;
      index_buffer[MONTH] = local_tm->tm_mon + 1;
      index_buffer[YEAR] = local_tm->tm_year;
      index_buffer[HIGH] = 0;
      index_buffer[LOW] = 0;
   }

   fd = create_file (index_filename);
   write_file (fd, INDEX_FILESIZE * sizeof(int), index_buffer);
   close_file (fd);

   h_increment_event_ctr ();
}

/*=======================================================================*
 *                           write_to_buffer                             *
 *=======================================================================*/
/* Buffer file header output.                                            */

PRIVATE
void write_to_buffer (fd, length, source)
int fd;
int length;
char far * source;
{
   char far * hdr;

   if (current_offset+length > BLOCK_SIZE) {
      write_file (fd, current_offset, ((int far *) header_buffer));
      current_offset = 0;
   }
   if (length >= BLOCK_SIZE) {
      write_file (fd, length, ((int far *) source));
   } else {
      hdr = &(header_buffer[current_offset]);
      movedata (FP_SEG(source), FP_OFF(source),
                FP_SEG(hdr), FP_OFF(hdr), length);
      current_offset += length;
   }
}

/*=======================================================================*
 *                             write_header                              *
 *=======================================================================*/
/* Construct and write a file header to new waveform file.               */

PRIVATE
void write_header (fd)
int fd;
{
   DT_28XX far * atodinfo;
   Q_CHANNEL far * triggers;
   Q_STATION far * stationcomp;
   T_EVENT far * eventsetting;
   T_TRIGGER far * trigsetting;

   current_offset = 0;

   set_starttime (TIMER_1);
   /* Write DETECTOR structure */
   write_to_buffer (fd, ((int)(file_header.structtag.len_struct +
                    sizeof(SUDS_STRUCTTAG))),
                    ((char far *) &(file_header.structtag)));

   /* Write ATODINFO structure */
   atodinfo = dt_get_atodinfo();
   write_to_buffer (fd, ((int)(atodinfo->structtag.len_struct +
                    sizeof(SUDS_STRUCTTAG))),
                    ((char far *) &(atodinfo->structtag)));

   /* Write TRIGSETTING structure */
   trigsetting = t_get_trigsetting();
   write_to_buffer (fd, ((int)(trigsetting->structtag.len_struct +
                    sizeof(SUDS_STRUCTTAG))),
                    ((char far *) &(trigsetting->structtag)));

   /* Write EVENTSETTING structure */
   eventsetting = t_get_eventsetting();
   write_to_buffer (fd,((int)(eventsetting->structtag.len_struct +
                    sizeof(SUDS_STRUCTTAG))),
                    ((char far *) &(eventsetting->structtag)));

   /* Write STATIONCOMP structures */
   stationcomp = st_get_head_station ();
   while (stationcomp != NULL) {
      write_to_buffer (fd, ((int)(stationcomp->structtag.len_struct +
                       sizeof(SUDS_STRUCTTAG))),
                       ((char far *) &(stationcomp->structtag)));
      stationcomp = st_get_next_station ();
   }

   /* Write TRIGGERS structures */
   triggers = t_get_head_trigger ();
   while (triggers != NULL) {
      write_to_buffer (fd, ((int)(triggers->structtag.len_struct +
                       sizeof(SUDS_STRUCTTAG))),
                       ((char far *) &(triggers->structtag)));
      triggers = t_get_next_trigger ();
   }

   /* Write the header_buffer to disk */
   if (current_offset) {
      write_file (fd, current_offset, ((int far *) header_buffer));
      current_offset = 0;
   }

   set_stoptime (TIMER_1);
   if (Debug_enabled)
      printf ("elapsed time to print header: %lf\n", diff_time(TIMER_1));
}

/*=======================================================================*
 *                            write_trailer                              *
 *=======================================================================*/
/* Construct and write a file trailer to new waveform file.              */

PRIVATE
void write_trailer (fd)
int fd;
{
   Q_ARRIVAL far * arrival;
   Q_LOCATION far * location;
   Q_MAGNITUDE far * magnitude;

   current_offset = 0;

   /* Write FEATURES structures */
   arrival = pk_get_first_arrival ();
   while (arrival != NULL) {
      write_to_buffer (fd, ((int)(arrival->structtag.len_struct +
                       sizeof(SUDS_STRUCTTAG))),
                       ((char far *) &(arrival->structtag)));
      arrival = pk_get_next_arrival ();
   }

   magnitude = pk_get_first_magnitude ();
   while (magnitude != NULL) {
      write_to_buffer (fd, ((int)(magnitude->structtag.len_struct +
                       sizeof(SUDS_STRUCTTAG))),
                       ((char far *) &(magnitude->structtag)));
      magnitude = pk_get_next_magnitude ();
   }

   /* Write ORIGIN structure if available */
   if ((location = l_get_location()) != NULL)
      write_to_buffer (fd,((int)(location->structtag.len_struct +
                       sizeof(SUDS_STRUCTTAG))),
                       ((char far *) &(location->structtag)));

   /* Write the header_buffer to disk */
   if (current_offset) {
      write_file (fd, current_offset, ((int far *) header_buffer));
      current_offset = 0;
   }
}

/*=======================================================================*
 *                           f_check_pathname                            *
 *=======================================================================*/
/* Check pathname for validity.                                          */

PUBLIC
void f_check_pathname ()
{
   long ds;
   int drive;

   /* Do we have any diskspace? */
   if ((ds = u_get_diskspace (path_name, &drive)) == 0)
      er_abort (F_DISK_FULL);
   else if (ds == INVALID_DRIVE)
      er_abort (F_INVALID_DRIVE);

   h_initialize_path (path_name);
}

/*=======================================================================*
 *                          f_get_event_number                           *
 *=======================================================================*/
/* Get current event number.                                             */

PUBLIC
long f_get_event_number ()
{
   return (event_number);
}

/*=======================================================================*
 *                         f_initialize_index                            *
 *=======================================================================*/
/* Open an existing index file or create a new one.  Start index counter
   where it left off in the existing index file or at zero if a new file
   is created.                                                           */

PUBLIC
void f_initialize_index ()
{
   struct tm * local_tm;
   long time;
   int fd;

   h_set_event_ctr ( 0);

   time = (long) (dm_get_current_time());
   local_tm = localtime (&time);
   build_index_fn();

   if ((fd = open_file (index_filename, READ)) != 0) {
      read_file (fd, INDEX_FILESIZE * sizeof(int), index_buffer);
      if ( index_buffer[DAY]   == local_tm->tm_mday  &&
           index_buffer[MONTH] == local_tm->tm_mon + 1   &&
           index_buffer[YEAR]  == local_tm->tm_year) {
         index.high = index_buffer[HIGH];
         index.low  = index_buffer[LOW];
         close_file (fd);
         return;
      }
   }
   b_init_base36 (&index);
   fd = create_file (index_filename);
   index_buffer[DAY] = local_tm->tm_mday;
   index_buffer[MONTH] = local_tm->tm_mon + 1;
   index_buffer[YEAR] = local_tm->tm_year;
   index_buffer[HIGH] = 0;
   index_buffer[LOW] = -1;
   write_file (fd, INDEX_FILESIZE * sizeof(int), index_buffer);
   close_file (fd);
}

/*=======================================================================*
 *                         f_initialize_params                           *
 *=======================================================================*/
/* Initialize key parameters.                                            */

PUBLIC
void f_initialize_params ()
{
   strcpy (path_name, PATH_NAME);

   suds_initialize (DETECTOR, &(file_header.info));
   suds_initialize_tag (DETECTOR, &(file_header.structtag));

   file_header.info.dalgorithm = 'x';
   file_header.info.versionnum = VERSIONNUM;
   u_strncpy (file_header.info.net_node_id, ((char far *) NETWORK_NODE_ID), 10);
}

/*=======================================================================*
 *                         f_set_NetworkNodeId                           *
 *=======================================================================*/
/* Set network node id.                                                  */

PUBLIC
void f_set_NetworkNodeId (id)
char far * id;
{
   u_strncpy (file_header.info.net_node_id, id, 10);
}

/*=======================================================================*
 *                            f_set_pathname                             *
 *=======================================================================*/
/* Set pathname.                                                         */

PUBLIC
void f_set_pathname (pn)
char pn[];
{
   unsigned int length;

   strncpy (path_name, pn, (MAX_FILENAME_LENGTH - 1));
   length = strlen(path_name);
   if (path_name[length-1] != ':' && path_name[length-1] != '\\')
      strcat (path_name, "\\");
}

/*=======================================================================*
 *                            f_get_pathname                             *
 *=======================================================================*/
/* Get pathname.                                                         */

PUBLIC
char * f_get_pathname ()
{
   return (path_name);
}

/*=======================================================================*
 *                            f_write_buffers                            *
 *=======================================================================*/
/* Write data buffer(s) to disk.                                         */

PUBLIC
void f_write_buffers (command, event_type)
unsigned int command;
char event_type;
{
   Q_BUFFER far * head;
   double abs_eventtime;
   int fd, drive;
   char out_str[80];

   switch (command) {
      case START_FILE:
         head = dm_get_head_buffer();
         switch (event_type) {
            case F_CALIBRATION:
               abs_eventtime = head->info.begintime;
               build_calib_fn (abs_eventtime);
               break;
            case F_EARTHQUAKE:
               update_index ();
               abs_eventtime = t_get_last_trigger();
               build_event_fn (abs_eventtime);
               break;
            case F_FREERUN:
            default:
               update_index ();
               abs_eventtime = head->info.begintime;
               build_event_fn (abs_eventtime);
         }
         file_header.info.event_type = event_type;
         file_header.info.event_number = event_number = build_event_number (abs_eventtime);

         fd = create_file (demux_filename);

         write_header (fd);

         if (event_type == F_FREERUN)
            head = dm_get_first_buffer();
         else head = dm_get_head_buffer();
         for (; head != NULL; head = dm_get_next_buffer()) {
            write_file (fd, head->structtag.len_struct + sizeof(SUDS_STRUCTTAG),
               ((int far *) &(head->structtag)));
            write_file (fd, (head->buffer_size * sizeof(unsigned)), head->data);
         }
         close_file (fd);
         break;
      case CONTINUE_FILE:
	 dsp_continue_band_recording();
         fd = open_file (demux_filename, WRITE);
         seek_file (fd, 0L, SEEK_END);
         for (head = dm_get_first_buffer(); head != NULL;
              head = dm_get_next_buffer()) {
            write_file (fd, head->structtag.len_struct + sizeof(SUDS_STRUCTTAG),
               ((int far *) &(head->structtag)));
            write_file (fd, (head->buffer_size * sizeof(unsigned)), head->data);
         }
         close_file (fd);
         break;
      case END_FILE:
         fd = open_file (demux_filename, WRITE);
         seek_file (fd, 0L, SEEK_END);
         write_trailer (fd);
         close_file (fd);

         sprintf (out_str, "\nEvent saved in: %s \n\n", demux_filename);
         if (Debug_enabled)
            printf (out_str);
         o_write_logfile (out_str);
         break;
   }

   if (u_get_diskspace(path_name, &drive) == 0)
      er_abort (F_DISK_FULL);
   h_update();
}
